package client;

import static org.easymock.EasyMock.*;
import java.util.Arrays;
import java.util.List;
import junit.framework.TestCase;
import common.IModel;
import common.IView;

public class PresenterTest extends TestCase {

	private IModel model;
	private IView view;
	private Presenter presenter;

	private List<String> searchOptions;

	@Override
	protected void setUp() {
		model = createStrictMock(IModel.class);
		view = createStrictMock(IView.class);
		presenter = new Presenter(view, model);
		view.initalizeAndDisplayView(presenter);

		searchOptions = Arrays.asList(presenter
				.getSearchOptionSelectionStrings());
	}

	private void finishSetup() {
		replay(model, view);
		presenter.displayViewOnScreen();
	}

	@Override
	protected void tearDown() throws Exception {
		verify(model, view);
	}

	public void testUserVisibleSearchSelectionOptions() {
		finishSetup();

		assertEquals(Arrays.asList("A only", "B only", "- ALL -"),
				searchOptions);
	}

	public void testAOnly() {
		expect(view.getSelectedSearchIndex()).andReturn(
				searchOptions.indexOf("A only"));
		expect(model.performSearch("A")).andReturn(
				Arrays.asList("some A value", "another A value"));
		view.setDisplayText("[some A value, another A value]");
		finishSetup();

		presenter.searchButtonPressed();
	}

	public void testBOnly() {
		expect(view.getSelectedSearchIndex()).andReturn(
				searchOptions.indexOf("B only"));
		expect(model.performSearch("B")).andReturn(
				Arrays.asList("some B value", "another B value"));
		view.setDisplayText("[some B value, another B value]");
		finishSetup();

		presenter.searchButtonPressed();
	}

	public void testAll() {
		expect(view.getSelectedSearchIndex()).andReturn(
				searchOptions.indexOf("- ALL -"));
		expect(model.performSearch("")).andReturn(
				Arrays.asList("A value", "B value", "C value"));
		view.setDisplayText("[A value, B value, C value]");
		finishSetup();

		presenter.searchButtonPressed();
	}

	/**
	 * From the perspective of the presenter, this should never happen. But if
	 * the model should experience a <code>RuntimeException</code> "for some
	 * inexplicable reason," ;-> we want the <code>Presenter</code> to display
	 * the exception class name.
	 */
	public void testModelThrowsRuntimeException() {
		expect(view.getSelectedSearchIndex()).andReturn(
				searchOptions.indexOf("A only"));
		expect(model.performSearch("A")).andThrow(new RuntimeException());
		view.setDisplayText(RuntimeException.class.getName());
		finishSetup();

		presenter.searchButtonPressed();
	}

    public void testAll2() {
        expect(view.getSelectedSearchIndex()).andReturn(searchOptions.indexOf("- ALL -"));
        expect(model.performSearch("")).andReturn(Arrays.asList("valueA", "valueB"));
        view.setDisplayText("[valueA, valueB]");
        finishSetup();

        presenter.searchButtonPressed();
    }
}
